<?php

/**
 * @file
 * Hooks provided by the features module.
 *
 * Features provides different kinds of hooks:
 *
 * - Regular module hooks.
 *   These are implemented by modules, like elsewhere in Drupal, and typically
 *   called via module_invoke_all().
 *   Most of these are to be implemented by contrib modules that define features
 *   components. Some implementations are provided by features itself, on behalf
 *   of other core or contrib modules.
 *
 * - Special "component hooks".
 *   These are implemented by features components, and usually invoked for one
 *   specific component only.
 *   For functions that implement these hooks, the "hook_" prefix is replaced
 *   with "{$component}_", where $component can be the name of the component, or
 *   its base component.
 *
 * - Generated feature module hooks.
 *   These contain the actual exported component data. Implementations are
 *   typically generated by features based on the values from
 *   hook_features_export_render().
 *   Some components already have these hooks defined in their own module, so
 *   they won't occur in this file.
 *
 * - Component data alter hooks.
 *   These can be implemented by any module, and allow to alter the exported
 *   component data from the generated hooks.
 *
 * - Feature module hooks.
 *   These can be implemented by feature modules, and typically called one at a
 *   time via module_invoke(), for the one feature module targeted by the
 *   current operation. Unlike the generated feature module hooks, these are
 *   typically added by a developer in the custom code part of a feature.
 *
 * @todo Rename component hooks as COMPONENT_features_xyz()? See #3077027.
 */

/**
 * Module hook. Allows a module to declare features components.
 *
 * @return array[]
 *   Format: $[$component] = $component_info
 *   An array of components, keyed by the component name. Each component can
 *   define several keys:
 *
 *   'file': Optional path to a file to include which contains the rest
 *   of the features API hooks for this module.
 *
 *   'default_hook': The defaults hook for your component that is called
 *   when the cache of default components is generated. Examples include
 *   hook_views_default_views() or hook_context_default_contexts().
 *
 *   'default_file': The file-writing behavior to use when exporting this
 *   component. May be one of 3 constant values:
 *
 *   FEATURES_DEFAULTS_INCLUDED_COMMON: write hooks/components to
 *   `.features.inc` with other components. This is the default behavior
 *   if this key is not defined.
 *
 *   FEATURES_DEFAULTS_INCLUDED: write hooks/components to a component-
 *   specific include named automatically by features.
 *
 *   FEATURES_DEFAULTS_CUSTOM: write hooks/components to a component-
 *   specific include with a custom name provided. If your module provides
 *   large amounts of code that should not be parsed often (only on specific
 *   cache clears/rebuilds, for example) you should use the 2nd or 3rd
 *   options to split your component into its own include.
 *
 *   'default_filename': The filename to use when 'default_file' is set to
 *   FEATURES_DEFAULTS_CUSTOM.
 *
 *   'feature_source': Boolean value for whether this component should be
 *   offered as an option on the initial feature creation form.
 *
 *   'base': Optional. An alternative base key to use when calling features
 *   hooks for this component. Can be used for features component types that
 *   are declared "dynamically" or are part of a family of components.
 *
 *   'alter_type': What type of alter hook this hook uses. 'normal' is called
 *   after the main hook is called. 'inline' is embedded within the default hook
 *   and may not be implemented by some default hooks.
 *   'none' is no alter hook exists. Defaults to 'normal'
 *
 *   'alter_hook': What the name of the alter hook for this component is.
 *    Do not include the '_alter' part. Defaults to 'default_hook'.
 */
function hook_features_api() {
  return array(
    'mycomponent' => array(
      'default_hook' => 'mycomponent_defaults',
      'default_file' => FEATURES_DEFAULTS_INCLUDED,
      'feature_source' => TRUE,
      'file' => drupal_get_path('module', 'mycomponent') . '/mycomponent.features.inc',
    ),
  );
}

/**
 * Component hook. Process the export array for a given component.
 *
 * The hook should be implemented using the name of the
 * component, not the module, eg. [component]_features_export() rather than
 * [module]_features_export().
 *
 * Implementations of this hook have three key tasks:
 *
 * 1. Determine module dependencies for any of the components passed to it
 *   e.g. the views implementation iterates over each views' handlers and
 *   plugins to determine which modules need to be added as dependencies.
 *
 * 2. Correctly add components to the export array. In general this is usually
 *   adding all of the items in $data to $export['features']['my_key'], but
 *   can become more complicated if components are shared between features
 *   or modules.
 *
 * 3. Delegating further detection and export tasks to related or derivative
 *   components.
 *
 * Each export processor can kickoff further export processors by returning a
 * keyed array (aka the "pipe") where the key is the next export processor hook
 * to call and the value is an array to be passed to that processor's $data
 * argument. This allows an export process to start simply at a few objects:
 *
 * [context]
 *
 * And then branch out, delegating each component to its appropriate hook:
 *
 * [context]--------+------------+
 *     |            |            |
 *   [node]      [block]      [views]
 *     |
 *   [CCK]
 *     |
 * [imagecache]
 *
 * @param string[] $data
 *   An array of machine names for the component in question to be exported.
 * @param array &$export
 *   By reference. An array of all components to be exported with a given
 *   feature. Component objects that should be exported should be added to
 *   this array.
 * @param string $module_name
 *   The name of the feature module to be generated.
 *
 * @return string[][]|void
 *   Format: $[$component][] = $name
 *   The pipe array of further processors that should be called.
 */
function hook_features_export($data, &$export, $module_name) {
  // The following is the simplest implementation of a straight object export
  // with no further export processors called.
  foreach ($data as $component) {
    $export['features']['mycomponent'][$component] = $component;
  }
  return array();
}

/**
 * Component hook. List all objects for a component that may be exported.
 *
 * The hook should be implemented using the name of the
 * component, not the module, eg. [component]_features_export() rather than
 * [module]_features_export().
 *
 * @return string[]
 *   Format: $[$value] = $label
 *   A keyed array of items, suitable for use with a FormAPI select or
 *   checkboxes element.
 */
function hook_features_export_options() {
  $options = array();
  foreach (mycomponent_load() as $mycomponent) {
    $options[$mycomponent->name] = $mycomponent->title;
  }
  return $options;
}

/**
 * Component hook. Renders one or more component objects to code.
 *
 * The hook should be implemented using the name of the component, not the
 * module, eg. [component]_features_export_render() rather than
 * [module]_features_export_render().
 *
 * @param string $module_name
 *   The name of the feature module to be exported.
 * @param array $data
 *   An array of machine name identifiers for the objects to be rendered.
 * @param array $export
 *   The full export array of the current feature being exported. This is only
 *   passed when hook_features_export_render() is invoked for an actual feature
 *   update or recreate, not during state checks or other operations.
 *
 * @return string[]|mixed[]
 *   Format: Combination of:
 *     - $[$hook] = $function_body
 *     - $[$hook] = ['code' => $function_body, 'args' => $params_php]
 *   An associative array of rendered PHP code where the key is the name of the
 *   hook that should wrap the PHP code. The hook should not include the name
 *   of the module, e.g. the key for `hook_example` should simply be `example`
 *   The values in the array can also be in the form of an associative array
 *   with the required key of 'code' and optional key of 'args', if 'args' need
 *   to be added to the hook. Alternate it can be an associative array in the
 *   same style as hook_features_export_files() to add additional files.
 */
function hook_features_export_render($module_name, $data, $export = NULL) {
  $code = array();
  $code[] = '$mycomponents = array();';
  foreach ($data as $name) {
    $code[] = "  \$mycomponents['{$name}'] = " . features_var_export(mycomponent_load($name)) . ";";
  }
  $code[] = "return \$mycomponents;";
  $code = implode("\n", $code);
  return array('mycomponent_defaults' => $code);
}

/**
 * Component hook. Reverts all component objects for a given feature module.
 *
 * The hook should be implemented using the name of the
 * component, not the module, eg. [component]_features_revert() rather than
 * [module]_features_export().
 *
 * @param string $module_name
 *   The name of the feature module whose components should be reverted.
 *
 * @return bool|void
 *   TRUE or FALSE for whether the components were successfully reverted.
 *   NOTE: This return value is no longer used in the latest Features so
 *   modules should no longer count on this value
 */
function hook_features_revert($module_name) {
  $mycomponents = module_invoke($module_name, 'mycomponent_defaults');
  if (!empty($mycomponents)) {
    foreach ($mycomponents as $mycomponent) {
      mycomponent_delete($mycomponent);
    }
  }
}

/**
 * Component hook. Rebuild all component objects for a given feature module.
 *
 * Should only be implemented for 'faux-exportable' components.
 *
 * The hook should be implemented using the name of the
 * component, not the module, eg. [component]_features_export() rather than
 * [module]_features_export().
 *
 * This hook is called at points where Features determines that it is safe
 * (ie. the feature is in state `FEATURES_REBUILDABLE`) for your module to
 * replace objects in the database with defaults that you collect from your
 * own defaults hook. See API.txt for how Features determines whether a
 * rebuild of components is possible.
 *
 * @param string $module_name
 *   The name of the feature module whose components should be rebuilt.
 */
function hook_features_rebuild($module_name) {
  $mycomponents = module_invoke($module_name, 'mycomponent_defaults');
  if (!empty($mycomponents)) {
    foreach ($mycomponents as $mycomponent) {
      mycomponent_save($mycomponent);
    }
  }
}

/**
 * Component hook. Called when a feature gets disabled.
 *
 * @param string $module_name
 *   Name of the feature that was disabled.
 *
 * @see \_features_restore()
 */
function hook_features_disable_feature($module_name) {
  // React to the feature being disabled.
}

/**
 * Component hook. Called when a feature gets enabled.
 *
 * @param string $module_name
 *   Name of the feature that was enabled.
 *
 * @see \_features_restore()
 */
function hook_features_enable_feature($module_name) {
  // React to the feature being enabled.
}

/**
 * Module hook. Invoked before a restore operation is run.
 *
 * This hook is called before any of the restore operations on the components is
 * run.
 *
 * @param string $op
 *   The operation that is triggered: revert, rebuild, disable, enable.
 * @param array $items
 *   The items handled by the operation.
 */
function hook_features_pre_restore($op, $items) {
  if ($op == 'rebuild') {
    // Use features rebuild to rebuild the features independent exports too.
    entity_defaults_rebuild();
  }
}

/**
 * Module hook. Invoked after a restore operation is run.
 *
 * This hook is called after any of the restore operations on the components is
 * run.
 *
 * @param string $op
 *   The operation that is triggered: revert, rebuild, disable, enable.
 * @param array $items
 *   The items handled by the operation.
 */
function hook_features_post_restore($op, $items) {
  if ($op == 'rebuild') {
    // Use features rebuild to rebuild the features independent exports too.
    entity_defaults_rebuild();
  }
}

/**
 * Module hook. Allows to alter the export options.
 *
 * The hook was first introduced in #1999254, to allow hiding specific items
 * that should never be exported to a feature.
 *
 * Known issues:
 * - This is currently ONLY invoked in the admin UI, NOT in drush commands.
 * - There are no implementations within features itself. In fact there might be
 *   no implementations of this hook in any contrib module.
 * - Use at your own risk! See #3079939.
 *
 * @param string[] $options
 *   Format: $[$name] = $label
 *   Machine names and labels of exportable items for the component.
 * @param string $component
 *   The component name, e.g. 'field_instance'.
 *
 * @see \hook_features_export_options()
 * @see \_features_export_build()
 */
function hook_features_export_options_alter(array &$options, $component) {
  // Alter the $options array.
}

/**
 * Module hook. Alter the final array of component names to be exported.
 *
 * Invoked just prior to the rendering of defaults. Allows modules a final say
 * in whether or not certain Components are exported (the Components' actual
 * data, however, cannot be altered by this hook).
 *
 * @param array &$export
 *   By reference. An array of all component names to be exported with a given
 *   feature.
 * @param array $module_name
 *   The name of the feature module to be generated.
 */
function hook_features_export_alter(&$export, $module_name) {
  // Example: do not allow the page content type to be exported, ever.
  if (!empty($export['features']['node']['page'])) {
    unset($export['features']['node']['page']);
  }
}

/**
 * Module hook. Alter the pipe array for a given component.
 *
 * This hook should be implemented with the name of the component type in place
 * of `component` in the function name, e.g. `features_pipe_views_alter()` will
 * alter the pipe for the Views component.
 *
 * @param string[][] $pipe
 *   Format: $[$component][] = $name
 *   By reference. The pipe array of further processors that should be called.
 * @param string[] $data
 *   An array of machine names for the component in question to be exported.
 * @param array $export
 *   An array of all components to be exported with a given feature.
 *   Some contrib implementations declare this parameter as by-reference, which
 *   allows them to modify it directly. Most implementations only modify the
 *   $pipe array.
 *   Special keys added to $export only during this alter hook:
 *   - $export['component'] (string), the component being exported.
 *   - $export['module_name'] (string), the module being exported to.
 */
function hook_features_pipe_COMPONENT_alter(&$pipe, $data, $export) {
  if (in_array($data, 'my-node-type')) {
    $pipe['dependencies'][] = 'mymodule';
  }
}

/**
 * Module hook. Alter the pipe array for a given component.
 *
 * @param string[][] &$pipe
 *   Format: $[$component][] = $name
 *   By reference. The pipe array of further processors that should be called.
 * @param string[] $data
 *   An array of machine names for the component in question to be exported.
 * @param array $export
 *   An array of all components to be exported with a given feature.
 *   Some contrib implementations declare this parameter as by-reference, which
 *   allows them to modify it directly. Most implementations only modify the
 *   $pipe array.
 *   Special keys added to $export only during this alter hook:
 *   - $export['component'] (string), the component being exported.
 *   - $export['module_name'] (string), the module being exported to.
 */
function hook_features_pipe_alter(&$pipe, $data, $export) {
  if ($export['component'] == 'node' && in_array('my-node-type', $data)) {
    $pipe['dependencies'][] = 'mymodule';
  }
}

/**
 * Module hook. Add extra files to the exported file.
 *
 * @param string $module_name
 *   Module being processed.
 * @param array $export
 *   Array of exported components.
 *
 * @return string[][]
 *   An array of files, keyed by file name that will appear in feature and
 *   with either file_path key to indicate where to copy the file from or
 *   file_content key to indicate the contents of the file.
 *   Format: A mix of:
 *   - $[$path]['file_content'] = $file_content
 *   - $[$path]['file_path'] = $source_file_to_copy
 */
function hook_features_export_files($module_name, $export) {
  return array('css/main.css' => array('file_content' => 'body {background-color:blue;}'));
}

/**
 * Module hook. Alter the extra files added to the export.
 *
 * @param array[] $files
 *   Format: A mix of:
 *   - $[$path]['file_content'] = $file_content
 *   - $[$path]['file_path'] = $source_file_to_copy.
 * @param string $module_name
 *   Name of the feature module being updated / generated.
 * @param array $export
 *   The export array, with a structure similar to a feature's *.info file.
 */
function hook_features_export_files_alter(&$files, $module_name, $export) {
  $files['css/main.css']['file_content'] = 'body {background-color:black;}';
}

/**
 * @defgroup features_component_alter_hooks Feature's component alter hooks
 * @{
 * Hooks to modify components defined by other features. These come in the form
 * hook_COMPONENT_alter where COMPONENT is the default_hook declared by any of
 * components within features.
 *
 * CTools also has a variety of hook_FOO_alters.
 *
 * Note: While views is a component of features, it declares its own alter
 * function which takes a similar form:
 * hook_views_default_views_alter(&$views)
 */

/**
 * Deprecated as of 7.x-2.0.
 *
 * Alter the default fields right before they are cached into the database.
 *
 * @param array[] &$fields
 *   By reference. The fields that have been declared by another feature.
 *
 * @deprecated
 */
function hook_field_default_fields_alter(&$fields) {
}

/**
 * Alter the base fields right before they are cached into the database.
 *
 * @param array[] &$fields
 *   Field base definitions from another feature, by field name.
 */
function hook_field_default_field_bases_alter(&$fields) {
}

/**
 * Alter the field instances right before they are cached into the database.
 *
 * @param array[] &$fields
 *   Format: $["$entity_type-$bundle-$field_name"] = $field_instance
 *   Field instance definitions from another feature.
 */
function hook_field_default_field_instances_alter(&$fields) {
}

/**
 * Alter default field groups right before they are cached into the database.
 *
 * @param array[] &$groups
 *   Field group definitions from another feature.
 *
 * @see \hook_field_group_info()
 * @see \hook_field_group_info_alter()
 */
function hook_fieldgroup_default_groups_alter(&$groups) {
}

/**
 * Alter default filter formats right before they are cached into the database.
 *
 * @param array[] &$formats
 *   By reference. The formats that have been declared by another feature.
 */
function hook_filter_default_formats_alter(&$formats) {
}

/**
 * Alter the default menus right before they are cached into the database.
 *
 * @param array[] &$menus
 *   By reference. The menus that have been declared by another feature.
 */
function hook_menu_default_menu_custom_alter(&$menus) {
}

/**
 * Alter the default menu links right before they are cached into the database.
 *
 * @param array[] &$links
 *   By reference. The menu links that have been declared by another feature.
 */
function hook_menu_default_menu_links_alter(&$links) {
}

/**
 * Alter the default menu items right before they are cached into the database.
 *
 * @param array[] &$items
 *   By reference. The menu items that have been declared by another feature.
 *
 * @deprecated
 * The 'menu' component should no longer be used.
 */
function hook_menu_default_items_alter(&$items) {
}

/**
 * Alter default vocabularies right before they are cached into the database.
 *
 * @param array[] &$vocabularies
 *   By reference. The vocabularies that have been declared by another feature.
 */
function hook_taxonomy_default_vocabularies_alter(&$vocabularies) {
}

/**
 * Alter default permissions right before they are cached into the database.
 *
 * @param array[] &$permissions
 *   By reference. The permissions that have been declared by another feature.
 */
function hook_user_default_permissions_alter(&$permissions) {
}

/**
 * Alter the default roles right before they are cached into the database.
 *
 * @param array[] &$roles
 *   By reference. The roles that have been declared by another feature.
 */
function hook_user_default_roles_alter(&$roles) {
}

/**
 * @}
 */


/**
 * @defgroup features_module_hooks Feature module hooks
 * @{
 * Hooks invoked on Feature modules when that module is enabled, disabled,
 * rebuilt, or reverted. These are ONLY invoked on the Features module on
 * which these actions are taken.
 */

/**
 * Feature module hook. Invoked on that module before it is reverted.
 *
 * @param string $component
 *   Name of the component that is about to be reverted.
 */
function hook_pre_features_revert($component) {
}

/**
 * Feature module hook. Invoked on that module after it is reverted.
 *
 * @param string $component
 *   Name of the component that has just been reverted.
 */
function hook_post_features_revert($component) {
}

/**
 * Feature module hook. Invoked on that module before it is rebuilt.
 *
 * @param string $component
 *   Name of the component that is about to be rebuilt.
 */
function hook_pre_features_rebuild($component) {
}

/**
 * Feature module hook. Invoked on that module after it is rebuilt.
 *
 * @param string $component
 *   Name of the component that has just been rebuilt.
 */
function hook_post_features_rebuild($component) {
}

/**
 * Feature module hook. Invoked on that module before it is disabled.
 *
 * @param string $component
 *   Name of the component that is about to be disabled.
 */
function hook_pre_features_disable_feature($component) {
}

/**
 * Feature module hook. Invoked on that module after it is disabled.
 *
 * @param string $component
 *   Name of the component that has just been disabled.
 */
function hook_post_features_disable_feature($component) {
}

/**
 * Feature module hook. Invoked on that module before it is enabled.
 *
 * @param string $component
 *   Name of the component that is about to be enabled.
 */
function hook_pre_features_enable_feature($component) {
}

/**
 * Feature module hook. Invoked on that module after it is enabled.
 *
 * @param string $component
 *   Name of the component that has just been enabled.
 */
function hook_post_features_enable_feature($component) {
}

/**
 * @}
 */
